<?php
/**
 * Markdown 解析器 (兼容 PHP 5.2+)
 * 将 Markdown 转换为 TailwindCSS 样式的 HTML
 */
class MarkdownParser {
    /**
     * 解析 Markdown 并生成 TailwindCSS 样式的 HTML
     * @param string $markdown Markdown 内容
     * @return string 生成的 HTML
     */
    public static function parse($markdown) {
        $lines = explode("\n", $markdown);
        $html = '';
        $inBlockquote = false;
        $inList = false;
        $inCodeBlock = false;
        $inTable = false;
        $tableRows = array();
        $tableAlignments = array();
        
        foreach ($lines as $line) {
            // 处理代码块
            if (preg_match('/^```/', $line)) {
                if ($inCodeBlock) {
                    $html .= '</code></pre>';
                    $inCodeBlock = false;
                } else {
                    $html .= '<pre class="bg-gray-800 text-gray-100 p-4 rounded-lg overflow-x-auto my-4"><code>';
                    $inCodeBlock = true;
                }
                continue;
            }
            
            if ($inCodeBlock) {
                $html .= htmlspecialchars($line) . "\n";
                continue;
            }
            
            // 处理表格
            if (strpos($line, '|') !== false && !$inTable) {
                // 检查是否是表头分隔线
                if (preg_match('/^[\|\-\:\s]+$/', str_replace(' ', '', $line))) {
                    $inTable = true;
                    $tableAlignments = self::parseTableAlignments($line);
                    continue;
                }
            }
            
            if ($inTable) {
                if (strpos($line, '|') === false) {
                    // 表格结束
                    $html .= self::generateTableHTML($tableRows, $tableAlignments);
                    $inTable = false;
                    $tableRows = array();
                    $tableAlignments = array();
                } else {
                    $tableRows[] = self::parseTableRow($line);
                    continue;
                }
            }
            
            // 处理标题
            if (preg_match('/^(#+)\s*(.*)/', $line, $matches)) {
                $level = strlen($matches[1]);
                $text = trim($matches[2]);
                $class = 'font-bold my-4';
                switch ($level) {
                    case 1: $class .= ' text-4xl'; break;
                    case 2: $class .= ' text-3xl'; break;
                    case 3: $class .= ' text-2xl'; break;
                    case 4: $class .= ' text-xl'; break;
                    case 5: $class .= ' text-lg'; break;
                    default: $class .= ' text-base';
                }
                $html .= sprintf('<h%d class="%s">%s</h%d>', $level, $class, self::parseInline($text), $level) . "\n";
                continue;
            }
            
            // 处理引用块
            if (preg_match('/^>\s*(.*)/', $line, $matches)) {
                if (!$inBlockquote) {
                    $html .= '<blockquote class="border-l-4 border-gray-400 pl-4 italic text-gray-600 my-4">';
                    $inBlockquote = true;
                }
                $html .= self::parseInline($matches[1]) . '<br/>';
                continue;
            } else if ($inBlockquote) {
                $html .= '</blockquote>';
                $inBlockquote = false;
            }
            
            // 处理列表
            if (preg_match('/^(\*|\-|\+)\s+(.*)/', $line, $matches) || preg_match('/^(\d+)\.\s+(.*)/', $line, $matches)) {
                $isOrdered = is_numeric($matches[1]);
                if (!$inList) {
                    $tag = $isOrdered ? 'ol' : 'ul';
                    $html .= sprintf('<%s class="list-%s pl-8 my-2 %s">', $tag, $isOrdered ? 'decimal' : 'disc', $isOrdered ? 'list-decimal' : 'list-disc');
                    $inList = true;
                }
                $html .= sprintf('<li class="my-1">%s</li>', self::parseInline($matches[2]));
                continue;
            } else if ($inList) {
                $html .= $inList ? '</' . (is_numeric(substr($html, strrpos($html, '<') + 2, 1)) ? 'ol' : 'ul') . '>' : '';
                $inList = false;
            }
            
            // 处理分隔线
            if (preg_match('/^(\-{3,}|\*{3,}|_{3,})$/', $line)) {
                $html .= '<hr class="my-4 border-t border-gray-300"/>';
                continue;
            }
            
            // 处理空行
            if (trim($line) === '') {
                if (strlen(trim($html)) > 0 && substr($html, -1) !== "\n") {
                    $html .= "\n";
                }
                continue;
            }
            
            // 处理普通段落
            $html .= sprintf('<p class="my-3 leading-relaxed">%s</p>', self::parseInline($line)) . "\n";
        }
        
        // 关闭任何未关闭的标签
        if ($inBlockquote) $html .= '</blockquote>';
        if ($inList) $html .= $inList ? '</' . (is_numeric(substr($html, strrpos($html, '<') + 2, 1)) ? 'ol' : 'ul') . '>' : '';
        if ($inCodeBlock) $html .= '</code></pre>';
        if ($inTable && !empty($tableRows)) {
            $html .= self::generateTableHTML($tableRows, $tableAlignments);
        }
        
        return $html;
    }
    
    /**
     * 解析行内元素
     * @param string $text 文本
     * @return string 处理后的 HTML
     */
    private static function parseInline($text) {
        // 处理图片
        $text = preg_replace_callback('/!\[(.*?)\]\((.*?)(?:\s+"(.*?)")?\)/', function($matches) {
            $alt = htmlspecialchars($matches[1]);
            $src = htmlspecialchars($matches[2]);
            $title = isset($matches[3]) ? ' title="' . htmlspecialchars($matches[3]) . '"' : '';
            return sprintf('<img src="%s" alt="%s"%s class="my-2 max-w-full h-auto rounded-lg shadow-md" />', $src, $alt, $title);
        }, $text);
        
        // 处理链接
        $text = preg_replace_callback('/\[(.*?)\]\((.*?)(?:\s+"(.*?)")?\)/', function($matches) {
            $title = isset($matches[3]) ? ' title="' . htmlspecialchars($matches[3]) . '"' : '';
            return sprintf('<a href="%s"%s class="text-blue-600 hover:text-blue-800 hover:underline">%s</a>', 
                htmlspecialchars($matches[2]), $title, self::parseInline($matches[1]));
        }, $text);
        
        // 处理行内代码
        $text = preg_replace('/`([^`]+)`/', '<code class="bg-gray-200 px-1 py-0.5 rounded text-sm font-mono">$1</code>', $text);
        
        // 处理加粗
        $text = preg_replace('/(\*\*|__)(.*?)\1/', '<strong class="font-bold">$2</strong>', $text);
        
        // 处理斜体
        $text = preg_replace('/(\*|_)(.*?)\1/', '<em class="italic">$2</em>', $text);
        
        // 处理删除线
        $text = preg_replace('/~~(.*?)~~/', '<del class="line-through">$1</del>', $text);
        
        return $text;
    }
    
    /**
     * 解析表格对齐方式
     * @param string $line 表格分隔线
     * @return array 对齐方式数组
     */
    private static function parseTableAlignments($line) {
        $alignments = array();
        $parts = explode('|', trim($line, '|'));
        
        foreach ($parts as $part) {
            $part = trim($part);
            if (preg_match('/^:?-+:?$/', $part)) {
                if ($part[0] === ':' && substr($part, -1) === ':') {
                    $alignments[] = 'center';
                } elseif ($part[0] === ':') {
                    $alignments[] = 'left';
                } elseif (substr($part, -1) === ':') {
                    $alignments[] = 'right';
                } else {
                    $alignments[] = 'left';
                }
            } else {
                $alignments[] = 'left';
            }
        }
        
        return $alignments;
    }
    
    /**
     * 解析表格行
     * @param string $line 表格行
     * @return array 单元格数组
     */
    private static function parseTableRow($line) {
        $cells = array();
        $parts = explode('|', trim($line, '|'));
        
        foreach ($parts as $part) {
            $cells[] = trim($part);
        }
        
        return $cells;
    }
    
    /**
     * 生成表格 HTML
     * @param array $rows 行数据
     * @param array $alignments 对齐方式
     * @return string 表格 HTML
     */
    private static function generateTableHTML($rows, $alignments) {
        if (empty($rows)) return '';
        
        $html = '<div class="overflow-x-auto my-4"><table class="min-w-full border-collapse">';
        $html .= '<thead><tr class="bg-gray-100">';
        
        // 表头
        foreach ($rows[0] as $i => $cell) {
            $align = isset($alignments[$i]) ? 'text-' . $alignments[$i] : 'text-left';
            $html .= sprintf('<th class="px-4 py-2 border border-gray-300 font-semibold %s">%s</th>', $align, self::parseInline($cell));
        }
        $html .= '</tr></thead><tbody>';
        
        // 表格内容
        for ($i = 1; $i < count($rows); $i++) {
            $html .= '<tr class="' . ($i % 2 === 0 ? 'bg-gray-50' : 'bg-white') . '">';
            foreach ($rows[$i] as $j => $cell) {
                $align = isset($alignments[$j]) ? 'text-' . $alignments[$j] : 'text-left';
                $html .= sprintf('<td class="px-4 py-2 border border-gray-300 %s">%s</td>', $align, self::parseInline($cell));
            }
            $html .= '</tr>';
        }
        
        $html .= '</tbody></table></div>';
        return $html;
    }
}

/**
 * 生成完整的 HTML 页面
 * @param string $title 页面标题
 * @param string $content HTML 内容
 * @return string 完整的 HTML
 */
function generateFullHTML($title, $content) {
    return sprintf('<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>%s</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <style>
        .prose {
            max-width: 1200;
        }
    </style>
</head>
<body class="bg-gray-50">
    <div class="container mx-auto px-4 py-8">
        <div class="prose mx-auto bg-white rounded-lg shadow-md p-8">
            %s
        </div>
    </div>
</body>
</html>', htmlspecialchars($title), $content);
}

// 使用示例
if (isset($_GET['file'])) {
    $file = $_GET['file'] ? trim($_GET['file']) : 'test.md';
    $markdown = file_get_contents($file);
} else {
    $markdown = '## 塑胶跑道专业施工流程详解  
 作者：湖北优正体育 (18571459135)

高品质塑胶跑道的诞生，依赖于严谨的施工流程与精细的工艺控制。湖北优正体育专注于塑胶运动场地建设，现将标准化施工流程解析如下：

---

### **一、 基础检视与处理**  
*   **检测标准：** 水泥/沥青基础需满足平整度（3m直尺误差≤3mm）、坡度（0.3%-0.8%）、强度（水泥C25以上，沥青压实度≥95%）及干燥度（含水率≤8%）要求。  
*   **缺陷修补：** 采用专用填缝材料处理伸缩缝，研磨凸起，填补孔洞、裂缝。  
*   **清洁除尘：** 彻底清除基础表面浮浆、油污、尘土，确保无杂质干扰粘结。

### **二、 防水底涂施工**  
*   **材料配比：** 严格按厂家指导比例混合单组份/双组份防水底涂。  
*   **涂布工艺：** 使用滚筒均匀滚涂，确保完全覆盖基础微孔，形成连续封闭膜（参考用量：0.2-0.3kg/㎡）。  
*   **固化要求：** 表干前禁止踩踏，环境温湿度达标后方可进入下道工序。

### **三、 弹性层摊铺**  
*   **配料精准：** 聚氨酯胶水+EPDM颗粒按设计比例机械搅拌，确保混合均匀无结团。  
*   **厚度控制：** 分层摊铺，采用摊铺机配合刮尺控制厚度（常见9-13mm），误差±1mm内。  
*   **密实平整：** 及时消泡、压实，接缝处做斜面处理，整体平整度达到≤3mm/3m。

### **四、 防滑面层喷涂**  
*   **双层喷涂工艺：**  
    *   **底层：** EPDM颗粒+聚氨酯喷涂料混合喷涂，形成纹理基底。  
    *   **面层：** 纯聚氨酯喷涂料+止滑粉罩面，提升耐磨抗紫外线性能。  
*   **参数控制：** 喷枪角度≥60°，压力恒定0.4-0.6MPa，喷涂方向正交，厚度≥2mm。

### **五、 精准划线**  
*   **定位放样：** 按国际田联（IAAF）或GB/T 14833标准，全站仪校准起跑线、道次等位置。  
*   **贴膜涂装：** 使用专业标线胶带定位，双组份聚氨酯划线漆涂刷2遍，确保线宽一致（5cm）、边缘锐利。

### **六、 科学养护管理**  
*   **固化期保护：** 施工后封闭养护≥7天，避免水浸、重压及尖锐物划伤。  
*   **环境监控：** 施工全程控制环境温度＞10℃、空气湿度＜85%，雨天禁止作业。

### **七、 质量检验与交付**  
*   **自检项目：** 覆盖平整度、厚度、坡度、色差、划线精度及物理性能（冲击吸收、垂直变形等）。  
*   **验收依据：** 参照《GB/T 14833-2020合成材料跑道面层》进行现场检测，提供完整施工记录。

---

**总结**  
塑胶跑道的高性能实现，核心在于对基础处理的严谨把控、分层施工的精细化操作、环境适配的科学养护以及严格的质量验收流程。湖北优正体育始终遵循标准化流程，致力于通过规范的工艺控制与专业的现场管理，为运动场地提供可靠的技术解决方案。';
}
// 解析 Markdown
$htmlContent = MarkdownParser::parse($markdown);

// 生成完整 HTML 页面
echo generateFullHTML('我的 Markdown 文档', $htmlContent);
?>