<?php
/**
 * SQLite Remote API - PHP 5.2 Compatible Version (Merged)
 * 
 * Supported Methods: GET, POST
 * Request Parameters:
 *   - db: SQLite database file path (required)
 *   - ac: query|exec|create (required)
 *   - sql: SQL statement to execute (required for query/exec)
 * 
 * Response: JSON format
 支持的三种操作

    创建数据库 (create)

        参数: db (数据库路径)

        示例: /dbs.php?ac=create&db=./my.d3

    查询数据 (query)

        参数: db, sql (SELECT语句)

        示例: /dbs.php?ac=query&db=./my.d3&sql=SELECT * FROM test
              /dbs.php?ac=query&db=./my.d3&sql=create table test(id integer primary key,title varchar(50))
              /dbs.php?ac=query&db=./my.d3&sql=create table test(id integer primary key,title varchar(50))
              /dbs.php?ac=query&db=./my.d3&sql=SELECT name FROM sqlite_master WHERE type='table' ORDER BY name

    执行SQL (exec)

        参数: db, sql (INSERT/UPDATE/DELETE等语句)

        示例: /dbs.php?ac=exec&db=./my.d3&sql=INSERT INTO test(title) VALUES('John')
        insert into test(title) values(\'Ajan\')
 */

// Set default headers
header('Content-Type: application/json');

// Check if SQLite is available
if (!extension_loaded('sqlite') && !extension_loaded('pdo_sqlite')) {
    die(json_encode(array(
        'status' => 'error',
        'message' => 'SQLite extension not available'
    )));
}

// Get request parameters (support both GET and POST)
$db_path = isset($_REQUEST['db']) ? $_REQUEST['db'] : null;
$ac = isset($_REQUEST['ac']) ? strtolower($_REQUEST['ac']) : null;
$sql = isset($_REQUEST['sql']) ? urldecode($_REQUEST['sql']) : null;
if (empty($db_path) && empty($ac)){
    die(json_encode(array(
        'status' => 'error',
        'message' => 'Unauthorized operation'
    )));
}
// Validate required parameters
if (empty($db_path)) {
    die(json_encode(array(
        'status' => 'error',
        'message' => 'Database path (db) is required'
    )));
}

if (empty($ac) || !in_array($ac, array('query', 'exec', 'create'))) {
    die(json_encode(array(
        'status' => 'error',
        'message' => 'Valid ac (query|exec|create) is required'
    )));
}

if ($ac != 'create' && empty($sql)) {
    die(json_encode(array(
        'status' => 'error',
        'message' => 'SQL statement is required for query/exec acs'
    )));
}

// Security check - prevent directory traversal and set base directory
$base_dir = './'; // Set your allowed directory here
$db_full_path = realpath(dirname($db_path)) . '/' . basename($db_path);

// Check if the target directory is within allowed base directory
if (strpos(realpath(dirname($db_path)), realpath($base_dir)) !== 0) {
    die(json_encode(array(
        'status' => 'error',
        'message' => 'Database path not allowed'
    )));
}

// Handle database creation
if ($ac == 'create') {
    // Check if database already exists
    if (file_exists($db_full_path)) {
        die(json_encode(array(
            'status' => 'error',
            'message' => 'Database already exists'
        )));
    }
    
    // Try to create the database
    try {
        if (extension_loaded('pdo_sqlite')) {
            $db = new PDO("sqlite:" . $db_full_path);
            $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $db = null; // Close connection
        } elseif (extension_loaded('sqlite')) {
            $db = sqlite_open($db_full_path, 0666, $error);
            if ($db) {
                sqlite_close($db);
            } else {
                throw new Exception($error);
            }
        }
        
        // Set appropriate permissions
        chmod($db_full_path, 0666);
        
        echo json_encode(array(
            'status' => 'success',
            'message' => 'Database created successfully',
            'path' => $db_full_path
        ));
        exit;
    } catch (Exception $e) {
        // Clean up if creation failed
        if (file_exists($db_full_path)) {
            unlink($db_full_path);
        }
        die(json_encode(array(
            'status' => 'error',
            'message' => 'Failed to create database: ' . $e->getMessage()
        )));
    }
}

// For query and exec acs, proceed with existing database
if (!file_exists($db_full_path)) {
    die(json_encode(array(
        'status' => 'error',
        'message' => 'Database file not found'
    )));
}

// Try to connect to SQLite database
$db = null;
$error = null;

if (extension_loaded('pdo_sqlite')) {
    // Try PDO first if available
    try {
        $db = new PDO("sqlite:" . $db_full_path);
        $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    } catch (PDOException $e) {
        $error = $e->getMessage();
    }
} elseif (extension_loaded('sqlite')) {
    // Fall back to SQLite if PDO not available
    $db = sqlite_open($db_full_path, 0666, $error);
}

if (!$db) {
    die(json_encode(array(
        'status' => 'error',
        'message' => 'Failed to connect to database: ' . $error
    )));
}

// Execute the requested ac
$result = null;
$success = false;

try {
    if ($ac == 'query') {
        // For SELECT queries
        if (is_a($db, 'PDO')) {
            $stmt = $db->query($sql);
            if ($stmt) {
                $result = $stmt->fetchAll(PDO::FETCH_ASSOC);
                $success = true;
            }
        } else {
            $result = sqlite_array_query($db, $sql, SQLITE_ASSOC);
            $success = $result !== false;
        }
    } elseif ($ac == 'exec') {
        // For INSERT, UPDATE, DELETE, etc.
        if (is_a($db, 'PDO')) {
            $affected = $db->exec($sql);
            if ($affected !== false) {
                $result = array('affected_rows' => $affected);
                $success = true;
            }
        } else {
            $success = sqlite_exec($db, $sql, $error);
            if ($success) {
                $result = array(
                    'affected_rows' => sqlite_changes($db)
                );
            }
        }
    }
    
    if (!$success) {
        if (is_a($db, 'PDO')) {
            $error = $db->errorInfo();
            $error = $error[2];
        } else {
            $error = sqlite_last_error($db) . ': ' . sqlite_error_string(sqlite_last_error($db));
        }
        
        throw new Exception($error);
    }
    
    echo json_encode(array(
        'status' => 'success',
        'message' => $sql,
        'data' => $result
    ));
    
} catch (Exception $e) {
    echo json_encode(array(
        'status' => 'error',
        'message' => $e->getMessage()
    ));
}

// Close connection
if (is_a($db, 'PDO')) {
    $db = null;
} else {
    sqlite_close($db);
}
?>