<?php
/**
 * RemoteDB Class
 * 
 * A simple database wrapper class compatible with PHP 5.2+
 * 
 * Usage:
 * $db = new RemoteDB('./my.db');
 * 
 * // Query
 * $result = $db->query("SELECT * FROM test WHERE id = ?", array('2'));
 * 
 * // Insert
 * $insertId = $db->exec("INSERT INTO test(title) VALUES(?)", array('John'));
 * 
 * // Update
 * $affectedRows = $db->exec("UPDATE test SET title = ? WHERE id = ?", array('Doe', '2'));
 * 
 * // Delete
 * $affectedRows = $db->exec("DELETE FROM test WHERE id = ?", array('2'));
 */
 
/*$db = new RemoteDB('./test.d3','https://atm-tkd.sakura.ne.jp/pcspe/access/');
$insertId = $db->exec("INSERT INTO test(title,key,desc) VALUES(?,?,?)", array(date("H:i:s"),date("H:i:s"),date("H:i:s")));
$result = $db->query("SELECT * FROM test", array());
print_r($result);*/
class RemoteDB {
    private $dbFile;
    private $Apiurl;
    /**
     * Constructor
     * 
     * @param string $dbFile Path to the SQLite database file
     */
    public function __construct($dbFile,$Apiurl) {
        $this->dbFile = $dbFile;
        $this->Apiurl = $Apiurl;
    }
    
    /**
     * Execute a query and return results
     * 
     * @param string $sql SQL query with ? placeholders
     * @param array $params Array of parameters to bind
     * @return array|false Array of results or false on failure
     */
    public function query($sql, $params = array()) {
        $url = $this->buildUrl('query', $sql, $params);
        $response = file_get_contents($url);
        
        if ($response === false) {
            return false;
        }
        
        $result = json_decode($response, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            return false;
        }
        
        return $result;
    }
    
    /**
     * Execute an INSERT, UPDATE, or DELETE statement
     * 
     * @param string $sql SQL statement with ? placeholders
     * @param array $params Array of parameters to bind
     * @return int|false Last insert ID (for INSERT) or affected rows count, false on failure
     */
    public function exec($sql, $params = array()) {
        $url = $this->buildUrl('exec', $sql, $params);
        $response = file_get_contents($url);
        
        if ($response === false) {
            return false;
        }
        
        $result = json_decode($response, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            return false;
        }
        
        return $result;
    }
    
    /**
     * Build the API URL with parameters
     * 
     * @param string $action 'query' or 'exec'
     * @param string $sql SQL statement
     * @param array $params Parameters to bind
     * @return string Encoded URL
     */
    private function buildUrl($action, $sql, $params) {
        // Encode parameters in the SQL statement
        $encodedSql = $this->prepareSql($sql, $params);
        
        // Build URL
        $url = $this->Apiurl . 'dbs.php?ac=' . urlencode($action) . 
               '&db=' . urlencode($this->dbFile) . 
               '&sql=' . urlencode($encodedSql);
        
        return $url;
    }
    
    /**
     * Prepare SQL statement by replacing placeholders with actual values
     * 
     * @param string $sql SQL with ? placeholders
     * @param array $params Parameters to bind
     * @return string Prepared SQL
     */
    private function prepareSql($sql, $params) {
        // Basic parameter escaping to prevent SQL injection
        foreach ($params as $param) {
            $param = str_replace("'", "''", $param); // Escape single quotes
            $param = "'" . $param . "'"; // Wrap in quotes
            
            // Replace the first ? with the parameter
            $sql = preg_replace('/\?/', $param, $sql, 1);
        }
        
        return $sql;
    }
}
?>