<?php
/**
 * 静态网页生成器
 * 兼容PHP5.2+，使用TailwindCSS创建响应式布局
 */

// 1. 定义内容数据 (可以是数据库查询结果或文件读取内容)
$contentData = array(
    'title' => '示例内容页面',
    'description' => '这是一个使用TailwindCSS生成的响应式网页示例',
    'sections' => array(
        array(
            'title' => '第一部分',
            'content' => '这是第一部分的内容，包含一些示例文本和说明。',
            'type' => 'text'
        ),
        array(
            'title' => '产品特性',
            'items' => array('响应式设计', '快速加载', '易于维护', '现代化界面'),
            'type' => 'list'
        ),
        array(
            'title' => '联系我们',
            'content' => '如有问题，请通过email@example.com联系我们',
            'type' => 'contact'
        )
    ),
    'meta' => array(
        'keywords' => 'PHP, TailwindCSS, 响应式设计',
        'author' => '网页生成器'
    )
);

// 2. 分析内容并生成HTML
function generateHtmlPage($data) {
    // 检测PHP版本以决定使用哪种JSON编码方式
    $jsonData = function_exists('json_encode') 
        ? json_encode($data) 
        : custom_json_encode($data);
    
    // 开始输出HTML
    $html = '<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>' . htmlspecialchars($data['title']) . '</title>
    <meta name="description" content="' . htmlspecialchars($data['description']) . '">
    <meta name="keywords" content="' . htmlspecialchars(isset($data['meta']['keywords']) ? $data['meta']['keywords'] : '' ) . '">
    <meta name="author" content="' . htmlspecialchars(isset($data['meta']['author']) ? $data['meta']['author'] : '' ) . '">
    
    <!-- 引入TailwindCSS CDN (使用兼容版本) -->
    <script src="https://cdn.tailwindcss.com/3.4.1"></script>
    
    <!-- 自定义交互脚本 -->
    <script>
    // 简单的交互功能
    document.addEventListener("DOMContentLoaded", function() {
        // 切换导航菜单 (移动端)
        var menuButton = document.getElementById("menu-button");
        var mobileMenu = document.getElementById("mobile-menu");
        
        if (menuButton && mobileMenu) {
            menuButton.addEventListener("click", function() {
                mobileMenu.classList.toggle("hidden");
            });
        }
        
        // 内容数据 (PHP传递过来的)
        var contentData = ' . $jsonData . ';
        console.log("页面内容数据:", contentData);
    });
    </script>
    
    <style>
    /* 兼容性样式补充 */
    .flex { display: -webkit-box; display: -ms-flexbox; display: flex; }
    @media (max-width: 640px) {
        .sm\\:hidden { display: none; }
    }
    </style>
</head>
<body class="bg-gray-100 font-sans leading-normal tracking-normal">
    
    <!-- 导航栏 -->
    <nav class="bg-white shadow-lg">
        <div class="container mx-auto px-4">
            <div class="flex justify-between items-center py-4">
                <div class="text-2xl font-semibold text-gray-800">
                    ' . htmlspecialchars($data['title']) . '
                </div>
                <!-- 移动端菜单按钮 -->
                <div class="md:hidden">
                    <button id="menu-button" class="text-gray-500 hover:text-gray-600 focus:outline-none">
                        <svg class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 12h16M4 18h16"></path>
                        </svg>
                    </button>
                </div>
                <!-- 桌面导航 -->
                <div class="hidden md:flex space-x-8">
                    <a href="#content" class="text-gray-700 hover:text-blue-600">内容</a>
                    <a href="#features" class="text-gray-700 hover:text-blue-600">特性</a>
                    <a href="#contact" class="text-gray-700 hover:text-blue-600">联系</a>
                </div>
            </div>
        </div>
        <!-- 移动端菜单 -->
        <div id="mobile-menu" class="hidden md:hidden bg-white pb-4 px-4">
            <a href="#content" class="block py-2 text-gray-700 hover:text-blue-600">内容</a>
            <a href="#features" class="block py-2 text-gray-700 hover:text-blue-600">特性</a>
            <a href="#contact" class="block py-2 text-gray-700 hover:text-blue-600">联系</a>
        </div>
    </nav>
    
    <!-- 主要内容区域 -->
    <!--main class="container mx-auto px-4 py-8"-->
    <main class="container mx-auto px-4 py-8 max-w-full md:max-w-2xl lg:max-w-4xl xl:max-w-6xl">
        <!-- 页面标题 -->
        <h1 class="text-3xl font-bold text-gray-800 mb-6">' . htmlspecialchars($data['title']) . '</h1>
        <p class="text-lg text-gray-600 mb-8">' . htmlspecialchars($data['description']) . '</p>
        
        <!-- 动态生成内容区块 -->';
        
    // 生成各个内容区块
    foreach ($data['sections'] as $section) {
        $html .= generateContentSection($section);
    }
    
    $html .= '
    </main>
    
    <!-- 页脚 -->
    <footer class="bg-gray-800 text-white py-8">
        <div class="container mx-auto px-4">
            <div class="flex flex-col md:flex-row justify-between items-center">
                <div class="mb-4 md:mb-0">
                    <p>&copy; ' . date('Y') . ' ' . htmlspecialchars($data['title']) . '. 保留所有权利.</p>
                </div>
                <div class="flex space-x-4">
                    <a href="#" class="hover:text-blue-400">隐私政策</a>
                    <a href="#" class="hover:text-blue-400">使用条款</a>
                </div>
            </div>
        </div>
    </footer>
</body>
</html>';
    
    return $html;
}

// 3. 内容区块生成函数
function generateContentSection($section) {
    $html = '
        <section class="mb-12 p-6 bg-white rounded-lg shadow-md">';
    
    // 根据内容类型生成不同的HTML结构
    switch ($section['type']) {
        case 'text':
            $html .= '
            <h2 class="text-2xl font-semibold text-gray-800 mb-4">' . htmlspecialchars($section['title']) . '</h2>
            <div class="prose max-w-none text-gray-700">
                ' . nl2br(htmlspecialchars($section['content'])) . '
            </div>';
            break;
            
        case 'list':
            $html .= '
            <h2 class="text-2xl font-semibold text-gray-800 mb-4">' . htmlspecialchars($section['title']) . '</h2>
            <ul class="list-disc pl-6 space-y-2">';
            
            foreach ($section['items'] as $item) {
                $html .= '
                <li class="text-gray-700">' . htmlspecialchars($item) . '</li>';
            }
            
            $html .= '
            </ul>';
            break;
            
        case 'contact':
            $html .= '
            <h2 class="text-2xl font-semibold text-gray-800 mb-4">' . htmlspecialchars($section['title']) . '</h2>
            <div class="text-gray-700">
                ' . nl2br(htmlspecialchars($section['content'])) . '
            </div>
            <form class="mt-4 space-y-4">
                <div>
                    <label for="name" class="block text-sm font-medium text-gray-700">姓名</label>
                    <input type="text" id="name" class="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 p-2 border">
                </div>
                <div>
                    <label for="email" class="block text-sm font-medium text-gray-700">邮箱</label>
                    <input type="email" id="email" class="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 p-2 border">
                </div>
                <div>
                    <label for="message" class="block text-sm font-medium text-gray-700">留言</label>
                    <textarea id="message" rows="4" class="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 p-2 border"></textarea>
                </div>
                <button type="button" onclick="alert(\'表单提交功能需JavaScript实现\')" class="inline-flex justify-center py-2 px-4 border border-transparent shadow-sm text-sm font-medium rounded-md text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                    提交
                </button>
            </form>';
            break;
            
        default:
            $html .= '
            <h2 class="text-2xl font-semibold text-gray-800 mb-4">' . htmlspecialchars($section['title']) . '</h2>
            <div class="text-gray-700">
                <p>未知内容类型</p>
            </div>';
    }
    
    $html .= '
        </section>';
    
    return $html;
}

// 4. PHP5.2兼容的JSON编码函数 (如果json_encode不可用)
if (!function_exists('json_encode')) {
    function custom_json_encode($data) {
        if (is_null($data)) return 'null';
        if ($data === false) return 'false';
        if ($data === true) return 'true';
        if (is_scalar($data)) {
            if (is_float($data)) {
                return floatval(str_replace(",", ".", strval($data)));
            }
            if (is_string($data)) {
                static $jsonReplaces = array(array("\\", "/", "\n", "\t", "\r", "\b", "\f", '"'), array('\\\\', '\\/', '\\n', '\\t', '\\r', '\\b', '\\f', '\"'));
                return '"' . str_replace($jsonReplaces[0], $jsonReplaces[1], $data) . '"';
            }
            return $data;
        }
        $isList = true;
        for ($i = 0, reset($data); $i < count($data); $i++, next($data)) {
            if (key($data) !== $i) {
                $isList = false;
                break;
            }
        }
        $result = array();
        if ($isList) {
            foreach ($data as $v) $result[] = custom_json_encode($v);
            return '[' . join(',', $result) . ']';
        } else {
            foreach ($data as $k => $v) $result[] = custom_json_encode($k).':'.custom_json_encode($v);
            return '{' . join(',', $result) . '}';
        }
    }
}

// 5. 生成并输出页面
header('Content-Type: text/html; charset=UTF-8');
echo generateHtmlPage($contentData);
?>