<?php
class TextReplacer {
    private $replacements = [];
    private $regexReplacements = [];
    private $dictionaryPath = '';
    
    /**
     * 构造函数
     * @param string $dictionaryPath 字典文件路径
     */
    public function __construct($dictionaryPath = '') {
        if ($dictionaryPath) {
            $this->setDictionaryPath($dictionaryPath);
        }
        
        // 预设一些常用替换规则
        $this->presetCommonRules();
    }
    
    /**
     * 设置字典文件路径
     * @param string $path 字典文件路径
     */
    public function setDictionaryPath($path) {
        if (file_exists($path)) {
            $this->dictionaryPath = $path;
            $this->loadDictionary();
        } else {
            throw new Exception("Dictionary file not found: " . $path);
        }
    }
    
    /**
     * 加载字典文件
     */
    private function loadDictionary() {
        $lines = file($this->dictionaryPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        
        foreach ($lines as $line) {
            // 跳过注释行
            if (strpos(trim($line), '#') === 0) {
                continue;
            }
            
            // 解析替换规则
            $parts = explode('=>', $line, 2);
            if (count($parts) === 2) {
                $search = trim($parts[0]);
                $replace = trim($parts[1]);
                
                // 检查是否是正则表达式 (以/开头和结尾)
                if (preg_match('/^\/.+\/[a-zA-Z]*$/', $search)) {
                    $this->addRegexReplacement($search, $replace);
                } else {
                    $this->addReplacement($search, $replace);
                }
            }
        }
    }
    
    /**
     * 添加简单替换规则
     * @param string $search  查找字符串
     * @param string $replace 替换字符串
     */
    public function addReplacement($search, $replace) {
        $this->replacements[$search] = $replace;
    }
    
    /**
     * 添加正则替换规则
     * @param string $pattern 正则模式
     * @param string $replace 替换字符串
     */
    public function addRegexReplacement($pattern, $replace) {
        $this->regexReplacements[$pattern] = $replace;
    }
    
    /**
     * 批量添加替换规则
     * @param array $rules 替换规则数组
     */
    public function addReplacements(array $rules) {
        foreach ($rules as $search => $replace) {
            if (preg_match('/^\/.+\/[a-zA-Z]*$/', $search)) {
                $this->addRegexReplacement($search, $replace);
            } else {
                $this->addReplacement($search, $replace);
            }
        }
    }
    
    /**
     * 预设一些常用替换规则
     */
    private function presetCommonRules() {
        // 常见拼写修正
        $this->addReplacement('teh', 'the');
        $this->addReplacement('adn', 'and');
        $this->addReplacement('thier', 'their');
        
        // 常见符号替换
        $this->addReplacement('&', 'and');
        $this->addReplacement('@', 'at');
        
        // 常见正则替换
        $this->addRegexReplacement('/\b(\d+)st\b/', '$1th');  // 1st -> 1th
        $this->addRegexReplacement('/\b(\d+)nd\b/', '$1th');  // 2nd -> 2th
        $this->addRegexReplacement('/\b(\d+)rd\b/', '$1th');  // 3rd -> 3th
        $this->addRegexReplacement('/\b(\d+)th\b/', '$1th');  // 保持一致性
    }
    
    /**
     * 执行替换
     * @param string $text 要处理的文本
     * @return string 处理后的文本
     */
    public function replace($text) {
        // 先执行简单替换
        if (!empty($this->replacements)) {
            $text = str_replace(
                array_keys($this->replacements),
                array_values($this->replacements),
                $text
            );
        }
        
        // 执行正则替换
        if (!empty($this->regexReplacements)) {
            foreach ($this->regexReplacements as $pattern => $replacement) {
                $text = preg_replace($pattern, $replacement, $text);
            }
        }
        
        return $text;
    }
    
    /**
     * 获取当前所有替换规则
     * @return array 包含所有替换规则的数组
     */
    public function getRules() {
        return [
            'simple' => $this->replacements,
            'regex' => $this->regexReplacements
        ];
    }
    
    /**
     * 清空所有替换规则
     */
    public function clearRules() {
        $this->replacements = [];
        $this->regexReplacements = [];
    }
}