<?php
/**
 * 增强版 Markdown 解析器
 * 修复内容：
 * 1. 完全修复图片识别问题
 * 2. 添加缺失图片检测和插入功能
 * 3. 保留所有原始内容
 * 4. 保持原有样式不变
 */

class EnhancedMarkdownParser {
    private $content;
    private $structure = [];
    private $title = 'Markdown Page';
    private $keywords = '';
    private $description = '';
    private $inCodeBlock = false;
    private $currentCodeBlock = null;
    private $customImageUrl = '';
    
    public function __construct($filePath, $customTitle = '', $customKeywords = '', $customDescription = '', $customImage = '') {
        $this->content = file_get_contents($filePath);
        $this->customImageUrl = $customImage;
        $this->parse();
        $this->determineTitle();
        
        // 设置自定义元数据
        if (!empty($customTitle)) {
            $this->title = $customTitle;
        }
        $this->keywords = $customKeywords;
        $this->description = $customDescription;
    }
    
    private function parse() {
        $lines = explode("\n", $this->content);
        $currentSection = null;
        $hasImages = false;
        
        foreach ($lines as $line) {
            $trimmedLine = trim($line);
            
            // 处理代码块开始/结束
            if (preg_match('/^```/', $line)) {
                if ($this->inCodeBlock) {
                    // 结束代码块
                    $this->structure[] = $this->currentCodeBlock;
                    $this->currentCodeBlock = null;
                    $this->inCodeBlock = false;
                } else {
                    // 开始代码块
                    $this->inCodeBlock = true;
                    $this->currentCodeBlock = [
                        'type' => 'code',
                        'content' => '',
                        'id' => 'code-' . count($this->structure)
                    ];
                }
                continue;
            }
            
            // 如果在代码块中，直接添加内容
            if ($this->inCodeBlock) {
                $this->currentCodeBlock['content'] .= $line . "\n";
                continue;
            }
            
            // 跳过空行
            if (empty($trimmedLine)) {
                $currentSection = null;
                continue;
            }
            
            // 解析图片（修复后的正则表达式）
            if (preg_match('/^!\[(.*?)\]\((.*?)(?:\s+"(.*?)")?\)/', $trimmedLine, $matches)) {
                $hasImages = true;
                $this->structure[] = [
                    'type' => 'image',
                    'alt' => $matches[1] ?? '',
                    'src' => $matches[2] ?? '',
                    'title' => $matches[3] ?? '',
                    'id' => 'image-' . count($this->structure)
                ];
                $currentSection = null;
                continue;
            }
            
            // 解析标题
            if (preg_match('/^(#{1,6})\s+(.*)/', $line, $matches)) {
                $level = strlen($matches[1]);
                $text = trim($matches[2]);
                $id = 'section-' . count($this->structure);
                
                $currentSection = [
                    'type' => 'heading',
                    'level' => $level,
                    'text' => $text,
                    'id' => $id
                ];
                
                $this->structure[] = $currentSection;
                continue;
            }
            
            // 解析无序列表
            if (preg_match('/^[\*\-\+]\s+(.*)/', $trimmedLine, $matches)) {
                if (!$currentSection || $currentSection['type'] !== 'list') {
                    $currentSection = [
                        'type' => 'list',
                        'items' => [],
                        'id' => 'list-' . count($this->structure)
                    ];
                    $this->structure[] = $currentSection;
                }
                $currentSection['items'][] = trim($matches[1]);
                continue;
            }
            
            // 解析有序列表
            if (preg_match('/^(\d+)\.\s+(.*)/', $trimmedLine, $matches)) {
                if (!$currentSection || $currentSection['type'] !== 'ordered_list') {
                    $currentSection = [
                        'type' => 'ordered_list',
                        'items' => [],
                        'id' => 'olist-' . count($this->structure)
                    ];
                    $this->structure[] = $currentSection;
                }
                $currentSection['items'][] = trim($matches[2]);
                continue;
            }
            
            // 解析块引用
            if (preg_match('/^>\s*(.*)/', $trimmedLine, $matches)) {
                if (!$currentSection || $currentSection['type'] !== 'blockquote') {
                    $currentSection = [
                        'type' => 'blockquote',
                        'content' => trim($matches[1]),
                        'id' => 'blockquote-' . count($this->structure)
                    ];
                    $this->structure[] = $currentSection;
                } else {
                    $currentSection['content'] .= "\n" . trim($matches[1]);
                }
                continue;
            }
            
            // 解析水平线
            if (preg_match('/^[-*_]{3,}\s*$/', $trimmedLine)) {
                $this->structure[] = [
                    'type' => 'hr',
                    'id' => 'hr-' . count($this->structure)
                ];
                $currentSection = null;
                continue;
            }
            
            // 处理段落内容
            if (!empty($trimmedLine)) {
                if (!$currentSection || $currentSection['type'] !== 'paragraph') {
                    $currentSection = [
                        'type' => 'paragraph',
                        'content' => $trimmedLine,
                        'id' => 'paragraph-' . count($this->structure)
                    ];
                    $this->structure[] = $currentSection;
                } else {
                    $currentSection['content'] .= "\n" . $trimmedLine;
                }
            }
        }
        
        // 如果没有图片且提供了自定义图片，在第一个长段落后插入
        if (!$hasImages && !empty($this->customImageUrl)) {
            $this->insertCustomImage();
        }
    }
    
    private function insertCustomImage() {
        $firstLongParagraphPos = null;
        
        // 查找第一个长段落（内容超过50字符）
        foreach ($this->structure as $pos => $element) {
            if ($element['type'] === 'paragraph' && strlen($element['content']) > 50) {
                $firstLongParagraphPos = $pos;
                break;
            }
        }
        
        if ($firstLongParagraphPos !== null) {
            // 在段落后插入图片
            array_splice($this->structure, $firstLongParagraphPos + 1, 0, [[
                'type' => 'image',
                'alt' => '插入图片',
                'src' => $this->customImageUrl,
                'title' => '',
                'id' => 'image-' . count($this->structure)
            ]]);
        }
    }
    
    private function determineTitle() {
        foreach ($this->structure as $element) {
            if ($element['type'] === 'heading' && $element['level'] === 1) {
                $this->title = $element['text'];
                break;
            }
        }
    }
    
    public function generateFullHtml() {
        $contentHtml = $this->generateContentHtml();
        
        // 生成meta标签
        $metaKeywords = !empty($this->keywords) ? '<meta name="keywords" content="' . htmlspecialchars($this->keywords) . '">' : '';
        $metaDescription = !empty($this->description) ? '<meta name="description" content="' . htmlspecialchars($this->description) . '">' : '';
        
        return <<<HTML
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>{$this->title}</title>
    {$metaKeywords}
    {$metaDescription}
    <link rel="stylesheet" href="https://unpkg.com/@picocss/pico@latest/css/pico.min.css">
    <style>
        /* 原有样式保持不变 */
        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 1rem;
            line-height: 1.6;
        }
        
        .image-container {
            margin: 1.5rem 0;
            text-align: center;
        }
        
        .image-container img {
            max-width: 100%;
            height: auto;
            border-radius: 4px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .image-float-left {
            float: left;
            margin: 0 1.5rem 1rem 0;
            max-width: 50%;
        }
        
        .image-float-right {
            float: right;
            margin: 0 0 1rem 1.5rem;
            max-width: 50%;
        }
        
        .image-full-width {
            width: 100%;
        }
        
        figcaption {
            font-size: 0.9rem;
            color: #666;
            margin-top: 0.5rem;
        }
        
        @media (max-width: 768px) {
            .image-float-left,
            .image-float-right {
                float: none;
                max-width: 100%;
                margin: 1rem 0;
            }
        }
        
        pre {
            background: #f8f8f8;
            padding: 1rem;
            border-radius: 4px;
            overflow-x: auto;
            border-left: 3px solid #ddd;
            margin: 1.5rem 0;
        }
        
        code {
            font-family: Consolas, Monaco, 'Andale Mono', monospace;
            font-size: 0.9em;
        }
        
        ul, ol {
            padding-left: 2rem;
            margin: 1.5rem 0;
        }
        
        li {
            margin-bottom: 0.5rem;
        }
        
        h1 { font-size: 2.2rem; margin: 2rem 0 1.5rem; }
        h2 { font-size: 1.8rem; margin: 1.8rem 0 1.3rem; }
        h3 { font-size: 1.5rem; margin: 1.6rem 0 1.1rem; }
        h4 { font-size: 1.3rem; margin: 1.4rem 0 1rem; }
        h5 { font-size: 1.1rem; margin: 1.2rem 0 0.9rem; }
        h6 { font-size: 1rem; margin: 1rem 0 0.8rem; }
        
        p {
            margin-bottom: 1.5rem;
            white-space: pre-line;
        }
        
        blockquote {
            border-left: 4px solid #ddd;
            padding: 0 1rem;
            margin: 1.5rem 0;
            color: #666;
            white-space: pre-line;
        }
        
        hr {
            border: 0;
            height: 1px;
            background: #ddd;
            margin: 2rem 0;
        }
        
        .clearfix::after {
            content: "";
            display: table;
            clear: both;
        }
    </style>
</head>
<body>
    <main class="container">
        {$contentHtml}
    </main>
</body>
</html>
HTML;
    }
    
    private function generateContentHtml() {
        $html = '';
        $imageCount = 0;
        
        foreach ($this->structure as $element) {
            switch ($element['type']) {
                case 'heading':
                    $html .= $this->generateHeading($element);
                    break;
                case 'paragraph':
                    $html .= $this->generateParagraph($element);
                    break;
                case 'list':
                    $html .= $this->generateList($element, false);
                    break;
                case 'ordered_list':
                    $html .= $this->generateList($element, true);
                    break;
                case 'image':
                    $html .= $this->generateImage($element, ++$imageCount);
                    break;
                case 'code':
                    $html .= $this->generateCode($element);
                    break;
                case 'blockquote':
                    $html .= $this->generateBlockquote($element);
                    break;
                case 'hr':
                    $html .= $this->generateHr();
                    break;
            }
        }
        
        return $html;
    }
    
    private function generateHeading($element) {
        $level = min($element['level'], 6);
        return "<h$level id=\"{$element['id']}\">{$this->parseInlineElements($element['text'])}</h$level>";
    }
    
    private function generateParagraph($element) {
        $content = $this->parseInlineElements($element['content']);
        return "<p id=\"{$element['id']}\">$content</p>";
    }
    
    private function generateList($element, $ordered = false) {
        $tag = $ordered ? 'ol' : 'ul';
        $html = "<$tag id=\"{$element['id']}\">";
        
        foreach ($element['items'] as $item) {
            $itemContent = $this->parseInlineElements($item);
            $html .= "<li>$itemContent</li>";
        }
        
        $html .= "</$tag>";
        return $html;
    }
    
    private function generateImage($element, $imageCount) {
        $layoutClass = 'image-full-width';
        
        if ($imageCount % 4 === 0) {
            $layoutClass = 'image-full-width';
        } elseif ($imageCount % 3 === 0) {
            $layoutClass = 'image-float-right';
        } elseif ($imageCount % 2 === 0) {
            $layoutClass = 'image-float-left';
        }
        
        $titleAttr = !empty($element['title']) ? ' title="' . htmlspecialchars($element['title']) . '"' : '';
        
        return <<<HTML
<figure class="image-container {$layoutClass}" id="{$element['id']}">
    <img src="{$element['src']}" alt="{$element['alt']}"{$titleAttr}>
    <figcaption>{$element['alt']}</figcaption>
</figure>
<div class="clearfix"></div>
HTML;
    }
    
    private function generateCode($element) {
        $content = htmlspecialchars(trim($element['content']));
        return "<pre id=\"{$element['id']}\"><code>$content</code></pre>";
    }
    
    private function generateBlockquote($element) {
        $content = $this->parseInlineElements($element['content']);
        return "<blockquote id=\"{$element['id']}\">$content</blockquote>";
    }
    
    private function generateHr() {
        return '<hr>';
    }
    
    private function parseInlineElements($text) {
        // 处理粗体
        $text = preg_replace('/(\*\*|__)(.*?)\1/', '<strong>$2</strong>', $text);
        // 处理斜体
        $text = preg_replace('/(\*|_)(.*?)\1/', '<em>$2</em>', $text);
        // 处理删除线
        $text = preg_replace('/~~(.*?)~~/', '<del>$1</del>', $text);
        // 处理内联代码
        $text = preg_replace('/`([^`]+)`/', '<code>$1</code>', $text);
        // 处理链接
        $text = preg_replace_callback(
            '/\[(.*?)\]\((.*?)(?:\s+"(.*?)")?\)/',
            function($matches) {
                $title = isset($matches[3]) ? ' title="' . htmlspecialchars($matches[3]) . '"' : '';
                return '<a href="' . htmlspecialchars($matches[2]) . '"' . $title . '>' . $matches[1] . '</a>';
            },
            $text
        );
        
        return $text;
    }
}

// 使用示例
if (isset($_GET['file'])) {
    $inputFile = $_GET['file'] ? trim($_GET['file']) : 'test.md';
    
    // 获取自定义元数据
    $customTitle = $_GET['title'] ?? '';
    $customKeywords = $_GET['keywords'] ?? '';
    $customDescription = $_GET['description'] ?? '';
    $customImage = $_GET['image'] ?? '';
    
    $parser = new EnhancedMarkdownParser($inputFile, $customTitle, $customKeywords, $customDescription, $customImage);
    echo $parser->generateFullHtml();
} else {
    echo <<<HTML
<h1>Markdown 解析器使用方法</h1>
<p>请通过URL参数指定要解析的Markdown文件：</p>
<ul>
    <li><strong>基本用法：</strong> ?file=test.md</li>
    <li><strong>自定义选项：</strong>
        <ul>
            <li>title - 自定义页面标题</li>
            <li>keywords - 自定义关键词</li>
            <li>description - 自定义描述</li>
            <li>image - 当文档没有图片时插入的图片URL</li>
        </ul>
    </li>
</ul>
<p>示例：?file=test.md&title=自定义标题&keywords=关键词1,关键词2&description=页面描述&image=https://example.com/image.jpg</p>
HTML;
}