<?php
/**
 * Simple Blog System using RemoteDB API - Corrected Version
 */
require (dirname(__file__) . '/RemoteDB.php');
// Initialize RemoteDB
$db = new RemoteDB('./blog.d3', 'https://atm-tkd.sakura.ne.jp/pcspe/access/');

// Initialize Blog System
$blog = new BlogSystem($db);

// Handle requests
$blog->handleRequest();

class BlogSystem {
    private $db;
    private $settings;
    private $prettyUrls = false;
    
    public function __construct($db) {
        $this->db = $db;
        $this->initializeDatabase();
        $this->loadSettings();
        $this->detectPrettyUrls();
    }
    
    private function initializeDatabase() {
        // Create settings table
        $this->db->exec("CREATE TABLE IF NOT EXISTS settings (
            id INTEGER PRIMARY KEY,
            name TEXT,
            value TEXT
        )");
        
        // Create categories table
        $this->db->exec("CREATE TABLE IF NOT EXISTS categories (
            id INTEGER PRIMARY KEY,
            parent_id INTEGER DEFAULT 0,
            name TEXT,
            slug TEXT UNIQUE,
            description TEXT,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
        )");
        
        // Create tags table
        $this->db->exec("CREATE TABLE IF NOT EXISTS tags (
            id INTEGER PRIMARY KEY,
            name TEXT,
            slug TEXT UNIQUE,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP
        )");
        
        // Create posts table
        $this->db->exec("CREATE TABLE IF NOT EXISTS posts (
            id INTEGER PRIMARY KEY,
            title TEXT,
            slug TEXT UNIQUE,
            content TEXT,
            excerpt TEXT,
            category_id INTEGER,
            status TEXT DEFAULT 'publish',
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            published_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY(category_id) REFERENCES categories(id)
        )");
        
        // Create post_tags junction table
        $this->db->exec("CREATE TABLE IF NOT EXISTS post_tags (
            post_id INTEGER,
            tag_id INTEGER,
            PRIMARY KEY(post_id, tag_id),
            FOREIGN KEY(post_id) REFERENCES posts(id),
            FOREIGN KEY(tag_id) REFERENCES tags(id)
        )");
    }
    
    private function loadSettings() {
        $results = $this->db->query("SELECT name, value FROM settings");
        $this->settings = [];
        #print_r($results);
        if ($results['data']) {
            foreach ($results['data'] as $row) {
                $this->settings[$row['name']] = $row['value'];
            }
        }
        
        // Set defaults if not exists
        $defaults = [
            'site_title' => 'My Blog',
            'site_description' => 'A simple blog system',
            'site_keywords' => 'blog, simple, php',
            'footer_text' => 'Powered by Simple Blog',
            'icp_number' => '',
            'pretty_urls' => '0'
        ];
        
        foreach ($defaults as $key => $value) {
            if (!isset($this->settings[$key])) {
                $this->settings[$key] = $value;
                $this->db->exec("INSERT INTO settings(name, value) VALUES(?, ?)", [$key, $value]);
            }
        }
    }
    
    private function detectPrettyUrls() {
        $this->prettyUrls = ($this->settings['pretty_urls'] === '1');
    }
    
    public function handleRequest() {
        $action = isset($_GET['action']) ? $this->sanitizeInput($_GET['action']) : '';
        $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
        
        // Admin actions
        if (isset($_GET['admin'])) {
            $this->checkAdminAuth();
            $this->handleAdminRequest($action, $id);
            return;
        }
        
        // Frontend actions
        switch ($action) {
            case 'post':
                $this->showPost($id);
                break;
            case 'category':
                $this->showCategory($id);
                break;
            case 'tag':
                $this->showTag($id);
                break;
            default:
                $this->showHomepage();
        }
    }
    
    private function handleAdminRequest($action, $id) {
        switch ($action) {
            case 'save_settings':
                $this->saveSettings();
                break;
            case 'toggle_pretty_urls':
                $this->togglePrettyUrls();
                break;
            case 'edit_category':
                $this->editCategory($id);
                break;
            case 'save_category':
                $this->saveCategory();
                break;
            case 'delete_category':
                $this->deleteCategory($id);
                break;
            case 'edit_post':
                $this->editPost($id);
                break;
            case 'save_post':
                $this->savePost();
                break;
            case 'delete_post':
                $this->deletePost($id);
                break;
            case 'edit_tag':
                $this->editTag($id);
                break;
            case 'save_tag':
                $this->saveTag();
                break;
            case 'delete_tag':
                $this->deleteTag($id);
                break;
            default:
                $this->showAdminDashboard();
        }
    }
    
    private function checkAdminAuth() {
        session_start();
        
        // Simple authentication - in a real app, use proper authentication
        if (!isset($_SESSION['admin_logged_in'])) {
            if (isset($_POST['username']) && isset($_POST['password']) && 
                $_POST['username'] === 'admin' && $_POST['password'] === 'admin123') {
                $_SESSION['admin_logged_in'] = true;
            } else {
                $this->showAdminLogin();
                exit;
            }
        }
    }
    
    private function showAdminLogin() {
        echo $this->renderTemplate('admin/login', [
            'site_title' => $this->settings['site_title']
        ]);
    }
    
    private function showAdminDashboard() {
        $posts = $this->db->query("SELECT * FROM posts ORDER BY created_at DESC LIMIT 5");
        $categories = $this->db->query("SELECT * FROM categories ORDER BY name");
        $tags = $this->db->query("SELECT * FROM tags ORDER BY name");
        
        echo $this->renderTemplate('admin/dashboard', [
            'settings' => $this->settings,
            'posts' => $posts ?: [],
            'categories' => $categories ?: [],
            'tags' => $tags ?: [],
            'prettyUrls' => $this->prettyUrls
        ]);
    }
    
    private function saveSettings() {
        foreach ($_POST as $name => $value) {
            if (array_key_exists($name, $this->settings)) {
                $safeValue = $this->sanitizeInput($value);
                $this->db->exec("UPDATE settings SET value = ? WHERE name = ?", [$safeValue, $name]);
                $this->settings[$name] = $safeValue;
            }
        }
        
        header("Location: ?admin");
        exit;
    }
    
    private function togglePrettyUrls() {
        $newValue = $this->prettyUrls ? '0' : '1';
        $this->db->exec("UPDATE settings SET value = ? WHERE name = 'pretty_urls'", [$newValue]);
        $this->settings['pretty_urls'] = $newValue;
        $this->prettyUrls = !$this->prettyUrls;
        
        header("Location: ?admin");
        exit;
    }
    
    private function editCategory($id = 0) {
        $category = $id ? $this->db->query("SELECT * FROM categories WHERE id = ?", [$id]) : null;
        $categories = $this->db->query("SELECT * FROM categories WHERE id != ? ORDER BY name", [$id]);
        
        echo $this->renderTemplate('admin/edit_category', [
            'category' => $category ? $category[0] : null,
            'categories' => $categories ?: [],
            'settings' => $this->settings
        ]);
    }
    
    private function saveCategory() {
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        $name = $this->sanitizeInput($_POST['name'] ?? '');
        $slug = $this->sanitizeInput($_POST['slug'] ?? '') ?: $this->generateSlug($name);
        $parentId = isset($_POST['parent_id']) ? (int)$_POST['parent_id'] : 0;
        $description = $this->sanitizeInput($_POST['description'] ?? '');
        
        if ($id) {
            $this->db->exec(
                "UPDATE categories SET name = ?, slug = ?, parent_id = ?, description = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?",
                [$name, $slug, $parentId, $description, $id]
            );
        } else {
            $this->db->exec(
                "INSERT INTO categories(name, slug, parent_id, description) VALUES(?, ?, ?, ?)",
                [$name, $slug, $parentId, $description]
            );
        }
        
        header("Location: ?admin");
        exit;
    }
    
    private function deleteCategory($id) {
        $this->db->exec("UPDATE posts SET category_id = 0 WHERE category_id = ?", [$id]);
        $this->db->exec("DELETE FROM categories WHERE id = ?", [$id]);
        
        header("Location: ?admin");
        exit;
    }
    
    private function editPost($id = 0) {
        $post = $id ? $this->db->query("SELECT * FROM posts WHERE id = ?", [$id]) : null;
        $categories = $this->db->query("SELECT * FROM categories ORDER BY name");
        $tags = $this->db->query("SELECT * FROM tags ORDER BY name");
        $postTags = $id ? $this->db->query("SELECT tag_id FROM post_tags WHERE post_id = ?", [$id]) : [];
        $postTagIds = $postTags ? array_column($postTags, 'tag_id') : [];
        
        echo $this->renderTemplate('admin/edit_post', [
            'post' => $post ? $post[0] : null,
            'categories' => $categories ?: [],
            'tags' => $tags ?: [],
            'postTagIds' => $postTagIds,
            'settings' => $this->settings
        ]);
    }
    
    private function savePost() {
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        $title = $this->sanitizeInput($_POST['title'] ?? '');
        $slug = $this->sanitizeInput($_POST['slug'] ?? '') ?: $this->generateSlug($title);
        $content = $this->sanitizeInput($_POST['content'] ?? '', true);
        $excerpt = $this->sanitizeInput($_POST['excerpt'] ?? '') ?: substr(strip_tags($content), 0, 200);
        $categoryId = isset($_POST['category_id']) ? (int)$_POST['category_id'] : 0;
        $status = $this->sanitizeInput($_POST['status'] ?? 'publish');
        $tags = isset($_POST['tags']) ? array_map('intval', $_POST['tags']) : [];
        
        if ($id) {
            $this->db->exec(
                "UPDATE posts SET title = ?, slug = ?, content = ?, excerpt = ?, category_id = ?, status = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?",
                [$title, $slug, $content, $excerpt, $categoryId, $status, $id]
            );
        } else {
            $this->db->exec(
                "INSERT INTO posts(title, slug, content, excerpt, category_id, status) VALUES(?, ?, ?, ?, ?, ?)",
                [$title, $slug, $content, $excerpt, $categoryId, $status]
            );
            $id = $this->db->query("SELECT last_insert_rowid() as id")[0]['id'];
        }
        
        // Update tags
        $this->db->exec("DELETE FROM post_tags WHERE post_id = ?", [$id]);
        foreach ($tags as $tagId) {
            $this->db->exec("INSERT INTO post_tags(post_id, tag_id) VALUES(?, ?)", [$id, $tagId]);
        }
        
        header("Location: ?admin");
        exit;
    }
    
    private function deletePost($id) {
        $this->db->exec("DELETE FROM posts WHERE id = ?", [$id]);
        $this->db->exec("DELETE FROM post_tags WHERE post_id = ?", [$id]);
        
        header("Location: ?admin");
        exit;
    }
    
    private function editTag($id = 0) {
        $tag = $id ? $this->db->query("SELECT * FROM tags WHERE id = ?", [$id]) : null;
        
        echo $this->renderTemplate('admin/edit_tag', [
            'tag' => $tag ? $tag[0] : null,
            'settings' => $this->settings
        ]);
    }
    
    private function saveTag() {
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        $name = $this->sanitizeInput($_POST['name'] ?? '');
        $slug = $this->sanitizeInput($_POST['slug'] ?? '') ?: $this->generateSlug($name);
        
        if ($id) {
            $this->db->exec(
                "UPDATE tags SET name = ?, slug = ? WHERE id = ?",
                [$name, $slug, $id]
            );
        } else {
            $this->db->exec(
                "INSERT INTO tags(name, slug) VALUES(?, ?)",
                [$name, $slug]
            );
        }
        
        header("Location: ?admin");
        exit;
    }
    
    private function deleteTag($id) {
        $this->db->exec("DELETE FROM tags WHERE id = ?", [$id]);
        $this->db->exec("DELETE FROM post_tags WHERE tag_id = ?", [$id]);
        
        header("Location: ?admin");
        exit;
    }
    
    private function showHomepage() {
        $posts = $this->db->query("SELECT * FROM posts WHERE status = 'publish' ORDER BY published_at DESC LIMIT 10");
        $categories = $this->db->query("SELECT * FROM categories ORDER BY name");
        
        echo $this->renderTemplate('home', [
            'posts' => $posts ?: [],
            'categories' => $categories ?: [],
            'settings' => $this->settings,
            'prettyUrls' => $this->prettyUrls
        ]);
    }
    
    private function showPost($idOrSlug) {
        if (is_numeric($idOrSlug)) {
            $post = $this->db->query("SELECT * FROM posts WHERE id = ?", [$idOrSlug]);
        } else {
            $post = $this->db->query("SELECT * FROM posts WHERE slug = ?", [$idOrSlug]);
        }
        
        if (!$post) {
            $this->show404();
            return;
        }
        
        $post = $post[0];
        $category = $post['category_id'] ? $this->db->query("SELECT * FROM categories WHERE id = ?", [$post['category_id']]) : null;
        $tags = $this->db->query("SELECT t.* FROM tags t JOIN post_tags pt ON t.id = pt.tag_id WHERE pt.post_id = ?", [$post['id']]);
        
        echo $this->renderTemplate('post', [
            'post' => $post,
            'category' => $category ? $category[0] : null,
            'tags' => $tags ?: [],
            'settings' => $this->settings,
            'prettyUrls' => $this->prettyUrls
        ]);
    }
    
    private function showCategory($idOrSlug) {
        if (is_numeric($idOrSlug)) {
            $category = $this->db->query("SELECT * FROM categories WHERE id = ?", [$idOrSlug]);
        } else {
            $category = $this->db->query("SELECT * FROM categories WHERE slug = ?", [$idOrSlug]);
        }
        
        if (!$category) {
            $this->show404();
            return;
        }
        
        $category = $category[0];
        $posts = $this->db->query("SELECT * FROM posts WHERE category_id = ? AND status = 'publish' ORDER BY published_at DESC", [$category['id']]);
        $categories = $this->db->query("SELECT * FROM categories ORDER BY name");
        
        echo $this->renderTemplate('category', [
            'category' => $category,
            'posts' => $posts ?: [],
            'categories' => $categories ?: [],
            'settings' => $this->settings,
            'prettyUrls' => $this->prettyUrls
        ]);
    }
    
    private function showTag($idOrSlug) {
        if (is_numeric($idOrSlug)) {
            $tag = $this->db->query("SELECT * FROM tags WHERE id = ?", [$idOrSlug]);
        } else {
            $tag = $this->db->query("SELECT * FROM tags WHERE slug = ?", [$idOrSlug]);
        }
        
        if (!$tag) {
            $this->show404();
            return;
        }
        
        $tag = $tag[0];
        $posts = $this->db->query("
            SELECT p.* 
            FROM posts p 
            JOIN post_tags pt ON p.id = pt.post_id 
            WHERE pt.tag_id = ? AND p.status = 'publish' 
            ORDER BY p.published_at DESC
        ", [$tag['id']]);
        
        $categories = $this->db->query("SELECT * FROM categories ORDER BY name");
        
        echo $this->renderTemplate('tag', [
            'tag' => $tag,
            'posts' => $posts ?: [],
            'categories' => $categories ?: [],
            'settings' => $this->settings,
            'prettyUrls' => $this->prettyUrls
        ]);
    }
    
    private function show404() {
        header("HTTP/1.0 404 Not Found");
        echo $this->renderTemplate('404', [
            'settings' => $this->settings
        ]);
    }
    
    private function generateSlug($text) {
        $slug = preg_replace('/[^a-z0-9]+/', '-', strtolower($text));
        $slug = trim($slug, '-');
        return $slug;
    }
    
    private function sanitizeInput($input, $allowHtml = false) {
        if ($allowHtml) {
            // Allow some basic HTML tags but still sanitize
            $input = strip_tags($input, '<p><a><strong><em><ul><ol><li><h1><h2><h3><h4><h5><h6><br><hr>');
            return htmlspecialchars($input, ENT_QUOTES, 'UTF-8');
        }
        return htmlspecialchars(strip_tags($input), ENT_QUOTES, 'UTF-8');
    }
    
    private function renderTemplate($template, $vars = []) {
        extract($vars);
        ob_start();
        
        // Template rendering logic here (same as before)
        // ...
        
        return ob_get_clean();
    }
}